<?php
require_once __DIR__ . '/../helpers.php';

class TripEngine {

    public static function haversineKm(float $lat1, float $lon1, float $lat2, float $lon2): float {
        $R = 6371.0;
        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);
        $a = sin($dLat/2)*sin($dLat/2) + cos(deg2rad($lat1))*cos(deg2rad($lat2))*sin($dLon/2)*sin($dLon/2);
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        return $R * $c;
    }

    // Smart ranking (minor-project friendly)
    public static function scorePlace(array $place, array $ctx): float {
        // ctx: start_lat, start_lng, interest_match (bool), rating (0..?), dist_km
        $score = 0;

        // Interest match weight
        $score += !empty($ctx['interest_match']) ? 30 : 0;

        // Rating (normalize: OTM rate is not standard 0..5. We'll clamp)
        if (!empty($place['rating'])) {
            $r = (float)$place['rating'];
            if ($r > 5) $r = 5;
            if ($r < 0) $r = 0;
            $score += ($r / 5.0) * 20.0;
        }

        // Distance penalty (simple)
        $dist = (float)($ctx['dist_km'] ?? 0);
        if ($dist <= 1) $score += 5;
        else if ($dist <= 3) $score += 2;
        else if ($dist <= 6) $score -= 5;
        else $score -= 12;

        // Popularity proxy: closer + rated => implicitly
        return $score;
    }

    public static function dedupeByExternalId(array $places): array {
        $seen = [];
        $out = [];
        foreach ($places as $p) {
            $id = $p['external_place_id'] ?? null;
            if (!$id) continue;
            if (isset($seen[$id])) continue;
            $seen[$id] = true;
            $out[] = $p;
        }
        return $out;
    }

    public static function nearestNeighborOrder(array $durations): array {
        // durations: NxN seconds, starting at index 0 (start point)
        $n = count($durations);
        $unvisited = [];
        for ($i = 1; $i < $n; $i++) $unvisited[$i] = true;

        $order = [0];
        $current = 0;
        while (!empty($unvisited)) {
            $best = null;
            $bestVal = null;
            foreach ($unvisited as $idx => $_) {
                $val = $durations[$current][$idx] ?? null;
                if ($val === null) continue;
                if ($bestVal === null || $val < $bestVal) {
                    $bestVal = $val;
                    $best = $idx;
                }
            }
            if ($best === null) break; // fallback
            $order[] = $best;
            unset($unvisited[$best]);
            $current = $best;
        }
        return $order;
    }

    public static function buildSchedule(string $startTime, string $endTime, array $routePlaceIds, int $visitMinutes, array $durationsSeconds, array $idxToPlaceId): array {
        // routePlaceIds includes only places (no start). idxToPlaceId maps matrix index -> place_id or 'START'
        $start = strtotime($startTime);
        $end = strtotime($endTime);

        $schedule = [];
        $cursor = $start;

        // route indices begins with 0 then ...
        $routeIdx = array_keys($idxToPlaceId); // not used
        // We'll compute schedule using order list on indices in idxToPlaceId
        // This function expects routePlaceIds in final visit order already.
        // We'll create a schedule list entries with travel + visit.

        $prevIdx = 0; // start index
        foreach ($routePlaceIds as $placeId) {
            // find index of placeId in idxToPlaceId
            $nextIdx = null;
            foreach ($idxToPlaceId as $i => $pid) {
                if ($pid === $placeId) { $nextIdx = $i; break; }
            }
            if ($nextIdx === null) continue;

            $travel = (int)round($durationsSeconds[$prevIdx][$nextIdx] ?? 0);
            $travelMin = (int)ceil($travel / 60);

            $travelStart = $cursor;
            $travelEnd = $cursor + ($travelMin * 60);

            // travel block (skip for first if user wants)
            if ($travelMin > 0) {
                $schedule[] = [
                    'type' => 'travel',
                    'from' => $idxToPlaceId[$prevIdx],
                    'to' => $placeId,
                    'start' => date('H:i', $travelStart),
                    'end' => date('H:i', $travelEnd),
                    'minutes' => $travelMin,
                ];
            }

            $visitStart = $travelEnd;
            $visitEnd = $visitStart + ($visitMinutes * 60);

            if ($visitEnd > $end) break;

            $schedule[] = [
                'type' => 'visit',
                'place_id' => $placeId,
                'start' => date('H:i', $visitStart),
                'end' => date('H:i', $visitEnd),
                'minutes' => $visitMinutes,
            ];
            $cursor = $visitEnd;
            $prevIdx = $nextIdx;
        }

        if ($cursor < $end) {
            $schedule[] = [
                'type' => 'buffer',
                'start' => date('H:i', $cursor),
                'end' => date('H:i', $end),
                'label' => 'Free time / buffer',
            ];
        }
        return $schedule;
    }

    public static function parseMoodToInterests(string $text): array {
        $t = strtolower($text);
        $map = [
            'calm' => ['nature', 'cafe'],
            'peace' => ['nature', 'temple'],
            'spiritual' => ['temple', 'historical'],
            'temple' => ['temple'],
            'food' => ['local_food', 'cafe'],
            'eat' => ['local_food', 'cafe'],
            'history' => ['historical', 'museum'],
            'museum' => ['museum'],
            'shopping' => ['shopping'],
            'adventure' => ['adventure', 'nature'],
            'night' => ['nightlife'],
            'party' => ['nightlife'],
            'nature' => ['nature'],
            'view' => ['nature'],
        ];
        $hits = [];
        foreach ($map as $k => $cats) {
            if (strpos($t, $k) !== false) {
                foreach ($cats as $c) $hits[$c] = true;
            }
        }
        return array_keys($hits);
    }
}
