<?php
require_once __DIR__ . '/config.php';

function start_session(): void {
    if (session_status() === PHP_SESSION_NONE) {
        session_name(SESSION_NAME);
        session_start([
            'cookie_httponly' => true,
            'cookie_samesite' => 'Lax',
        ]);
    }
}

function e(string $s): string {
    return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

function redirect(string $path): void {
    header('Location: ' . BASE_URL . $path);
    exit;
}

function set_flash(string $type, string $msg): void {
    start_session();
    $_SESSION['flash'] = ['type' => $type, 'msg' => $msg];
}

function get_flash(): ?array {
    start_session();
    if (!isset($_SESSION['flash'])) return null;
    $f = $_SESSION['flash'];
    unset($_SESSION['flash']);
    return $f;
}

function csrf_token(): string {
    start_session();
    if (empty($_SESSION[CSRF_KEY])) {
        $_SESSION[CSRF_KEY] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_KEY];
}

function csrf_verify(): void {
    start_session();
    $sent = $_POST[CSRF_KEY] ?? '';
    $real = $_SESSION[CSRF_KEY] ?? '';
    if (!$real || !$sent || !hash_equals($real, $sent)) {
        http_response_code(403);
        die('CSRF verification failed.');
    }
}

function require_auth(): void {
    start_session();
    if (empty($_SESSION['user_id'])) {
        set_flash('warning', 'Please login to continue.');
        redirect('/auth/login.php');
    }
}

function require_admin(): void {
    start_session();
    if (empty($_SESSION['admin_id'])) {
        set_flash('warning', 'Admin login required.');
        redirect('/admin/login.php');
    }
}

function post_only(): void {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        die('Method Not Allowed');
    }
}

function validate_email(string $email): bool {
    return (bool)filter_var($email, FILTER_VALIDATE_EMAIL);
}

function now(): string {
    return date('Y-m-d H:i:s');
}

function http_json(string $url, array $headers = [], int $timeout = 15): array {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_CONNECTTIMEOUT => 8,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_USERAGENT => 'SmartTouristGuide/1.0 (+educational-project)',
    ]);
    $raw = curl_exec($ch);
    $err = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($raw === false) {
        throw new Exception('HTTP request failed: ' . $err);
    }
    if ($code >= 400) {
        throw new Exception('HTTP error ' . $code . ' for URL: ' . $url);
    }

    $data = json_decode($raw, true);
    if ($data === null) {
        throw new Exception('Invalid JSON response from: ' . $url);
    }
    return $data;
}

function ensure_storage_dirs(): void {
    $dirs = [
        __DIR__ . '/../storage',
        __DIR__ . '/../storage/cache',
        __DIR__ . '/../storage/logs',
    ];
    foreach ($dirs as $d) {
        if (!is_dir($d)) @mkdir($d, 0775, true);
    }
}
